/*  
 * Copyright (C) 2005 Martin Pischky (mailto:martin@pischky.de)  
 *
 * This file (ChunkedMemory.java) is part of IntelHexFile.
 *
 * IntelHexFile is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 *
 */

/*
 * $Log: ChunkedMemory.java,v $
 * Revision 1.1  2005/03/27 12:29:58  pischky
 * added
 *
 */

package org.fremo.ihx;

/**
 * @author martin
 */
public class ChunkedMemory 
extends AbstractMemory 
{
    
    private static int CHUNK_SIZE = 1024;
    private static int MAX_CHUNK = (ADDRESS_HIGH+1) / CHUNK_SIZE - 1;
    private static String NEW_LINE = System.getProperty("line.separator");
    
   	protected class Chunk {
        
        int base;
   		byte[] values = new byte[CHUNK_SIZE];
        boolean[] defined = new boolean[CHUNK_SIZE];
        
        Chunk( int base ) {
            this.base = base/CHUNK_SIZE*CHUNK_SIZE;
        }
        
        public String toString() {
            StringBuffer b = new StringBuffer( 2048 );
            for (int i = 0; i < CHUNK_SIZE/16; i++) {
                boolean def = false;
                for (int j = 0; j < 16; j++) {
                    def |= defined[16*i+j];
                }
                if( def ) {
                    b.append( ""+HexUtils.hexString4(base+16*i)+"  " );
                    for (int j = 0; j < 16; j++) {
                        if( defined[16*i+j] ) {
                            b.append(HexUtils.hexString2(values[16*i+j])+" ");
                        } else {
                            b.append("-- ");
                        }
                        if( j==7 ) b.append(" ");
                    }
                    b.append(NEW_LINE);
                }
            }
            return b.toString();
        }
	
    }
    
    private Chunk[] chunks = new Chunk[ MAX_CHUNK+1 ];
    
	/**
	 * New ChunkedMemory where all addresse are "UNDEFINED" 
	 */
	public ChunkedMemory() {
		super();
	}
    
    /* (non-Javadoc)
	 * @see org.fremo.ihx.Memory#set(int, byte)
	 */
	public void set(int address, byte value) 
    throws InvalidAddressException
    {
        AbstractMemory.checkAddress(address);
		int chunkNum = address / CHUNK_SIZE;
        int chunkAdr = address % CHUNK_SIZE;
        if( chunks[chunkNum] == null ) {
        	chunks[chunkNum] = new Chunk( address );
        }
        chunks[chunkNum].values[chunkAdr] = value;
        chunks[chunkNum].defined[chunkAdr] = true;
	}

    /* (non-Javadoc)
	 * @see org.fremo.ihx.Memory#get(int)
	 */
	public short get(int address) 
    throws InvalidAddressException
    {
        AbstractMemory.checkAddress(address);
        int chunkNum = address / CHUNK_SIZE;
        int chunkAdr = address % CHUNK_SIZE;
        if( chunks[chunkNum] == null ) {
            return UNDEFINED;
        }
        if( ! chunks[chunkNum].defined[chunkAdr] ) {
            return UNDEFINED;           
        }
        short val = chunks[chunkNum].values[chunkAdr];
        if( val < 0 ) val = (short) (256 + val);
        return val;
	}

	/* (non-Javadoc)
	 * @see org.fremo.ihx.Memory#isDefined(int)
	 */
	public boolean isDefined(int address) 
    throws InvalidAddressException
    {
        AbstractMemory.checkAddress(address);
        int chunkNum = address / CHUNK_SIZE;
        int chunkAdr = address % CHUNK_SIZE;
        if( chunks[chunkNum] == null ) {
            return false;
        }
        return chunks[chunkNum].defined[chunkAdr];           
	}

    /* (non-Javadoc)
     * @see org.fremo.ihx.Memory#undefine(int)
     */
    public void undefine(int address) 
    throws InvalidAddressException 
    {
        AbstractMemory.checkAddress(address);
        int chunkNum = address / CHUNK_SIZE;
        int chunkAdr = address % CHUNK_SIZE;
        if( chunks[chunkNum] == null ) {
            return;
        }
        chunks[chunkNum].defined[chunkAdr] = false;
        //TODO: if chunk is completly unused now, dispose it
    }
    
    //TODO: implement a faster version of nextUsedAddress
    
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString() {
        StringBuffer b = new StringBuffer( 2048 );
        for (int i = 0; i < chunks.length; i++) {
            if( chunks[i] != null ) {
                b.append( chunks[i].toString() );
            }
        } 
        return b.toString();
    }
}
